"use client";
import type { ReactNode } from "react";
import Link from "next/link";

type ButtonProps = {
  children: ReactNode;
  variant?:
    | "primary"
    | "secondary"
    | "outline"
    | "ghost"
    | "gradient"
    | "customPurple";
  size?: "sm" | "md" | "lg";
  className?: string;
  onClick?: () => void;
  href?: string;
  type?: "button" | "submit" | "reset";
  disabled?: boolean;
  icon?: ReactNode;
  style?: React.CSSProperties;
};

const Button = ({
  children,
  variant = "primary",
  size = "md",
  className = "",
  onClick,
  href,
  type = "button",
  disabled = false,
  icon,
}: ButtonProps) => {
  const baseClasses =
    "rounded-lg font-semibold transition-all duration-200 flex items-center justify-center gap-2 cursor-pointer";

  const sizeClasses = {
    sm: "px-3 py-1.5 text-sm",
    md: "px-4 py-2 text-base",
    lg: "px-6 py-3 text-lg",
  };

  const variantClasses = {
    primary:
      "bg-[#7847E3] text-white hover:bg-[#6a3ece] shadow hover:shadow-md",
    secondary: "bg-gray-100 text-black hover:text-[#7847E3]",
    outline: "border border-[#7847E3] text-[#7847E3] hover:bg-[#7847E3]/10",
    ghost: "text-[#7847E3] hover:bg-[#7847E3]/10",
    gradient:
      "bg-gradient-to-r from-purple-600 to-indigo-600 text-white hover:from-purple-700 hover:to-indigo-700 shadow hover:shadow-md",
    customPurple:
      "border border-[#7847E3] bg-[#7847E3] text-white hover:bg-[#6a3ece]",
  };

  const disabledClasses = "opacity-50 cursor-not-allowed";

  const combinedClasses = `
    ${baseClasses}
    ${sizeClasses[size]}
    ${variantClasses[variant]}
    ${disabled ? disabledClasses : ""}
    ${className}
  `;

  if (href) {
    return (
      <Link href={href} className={combinedClasses}>
        {icon && <span>{icon}</span>}
        {children}
      </Link>
    );
  }

  return (
    <button
      type={type}
      className={combinedClasses}
      onClick={onClick}
      disabled={disabled}
    >
      {icon && <span>{icon}</span>}
      {children}
    </button>
  );
};

export default Button;
